# Print an optspec for argparse to handle cmd's options that are independent of any subcommand.
function __fish_maturin_global_optspecs
	string join \n v/verbose h/help V/version
end

function __fish_maturin_needs_command
	# Figure out if the current invocation already has a command.
	set -l cmd (commandline -opc)
	set -e cmd[1]
	argparse -s (__fish_maturin_global_optspecs) -- $cmd 2>/dev/null
	or return
	if set -q argv[1]
		# Also print the command, so this can be used to figure out what it is.
		echo $argv[1]
		return 1
	end
	return 0
end

function __fish_maturin_using_subcommand
	set -l cmd (__fish_maturin_needs_command)
	test -z "$cmd"
	and return 1
	contains -- $cmd[1] $argv
end

complete -c maturin -n "__fish_maturin_needs_command" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_needs_command" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_needs_command" -s V -l version -d 'Print version'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "build" -d 'Build the crate into python packages'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "publish" -d 'Build and publish the crate as python packages to pypi'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "list-python" -d 'Search and list the available python installations'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "develop" -d 'Install the crate as module in the current virtualenv'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "sdist" -d 'Build only a source distribution (sdist) without compiling'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "init" -d 'Create a new cargo project in an existing directory'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "new" -d 'Create a new cargo project'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "generate-ci" -d 'Generate CI configuration'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "upload" -d 'Upload python packages to pypi'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "pep517" -d 'Backend for the PEP 517 integration. Not for human consumption'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "completions" -d 'Generate shell completions'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "zig" -d 'Zig linker wrapper'
complete -c maturin -n "__fish_maturin_needs_command" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l strip -d 'Strip the library for minimum file size. Can be set to `true` or `false`, or used as a flag (`--strip` implies `true`)' -r -f -a "true\t''
false\t''"
complete -c maturin -n "__fish_maturin_using_subcommand build" -l compatibility -d 'Control the platform tag and PyPI compatibility' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -s i -l interpreter -d 'The python versions to build wheels for, given as the executables of interpreters such as `python3.9` or `/usr/bin/python3.8`' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand build" -s b -l bindings -d 'Which kind of bindings to use' -r -f -a "pyo3\t''
pyo3-ffi\t''
cffi\t''
uniffi\t''
bin\t''"
complete -c maturin -n "__fish_maturin_using_subcommand build" -s o -l out -d 'The directory to store the built wheels in. Defaults to a new "wheels" directory in the project\'s target directory' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand build" -l auditwheel -d 'Audit wheel for manylinux compliance' -r -f -a "repair\t'Audit and repair wheel for manylinux compliance'
check\t'Check wheel for manylinux compliance, but do not repair'
skip\t'Don\'t check for manylinux compliance'"
complete -c maturin -n "__fish_maturin_using_subcommand build" -s j -l jobs -d 'Number of parallel jobs, defaults to # of CPUs' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -l profile -d 'Build artifacts with the specified Cargo profile' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -s F -l features -d 'Space or comma separated list of features to activate' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -l target -d 'Build for the target triple' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -l target-dir -d 'Directory for all generated artifacts' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand build" -s m -l manifest-path -d 'Path to Cargo.toml' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand build" -l color -d 'Coloring: auto, always, never' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -l config -d 'Override a configuration value (unstable)' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -s Z -d 'Unstable (nightly-only) flags to Cargo, see \'cargo -Z help\' for details' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -l timings -d 'Timing output formats (unstable) (comma separated): html, json' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -l sbom-include -d 'Additional SBOM files to include in the `.dist-info/sboms` directory. Can be specified multiple times' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand build" -l compression-method -d 'Zip compression method. Only Stored and Deflated are currently compatible with all package managers' -r -f -a "deflated\t'Deflate compression (levels 0-9, default 6)'
stored\t'No compression'
bzip2\t'BZIP2 compression (levels 0-9, default 6)'
zstd\t'Zstandard compression (supported from Python 3.14; levels -7-22, default 3)'"
complete -c maturin -n "__fish_maturin_using_subcommand build" -l compression-level -d 'Zip compression level. Defaults to method default' -r
complete -c maturin -n "__fish_maturin_using_subcommand build" -s r -l release -d 'Build artifacts in release mode, with optimizations'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l sdist -d 'Build a source distribution'
complete -c maturin -n "__fish_maturin_using_subcommand build" -s f -l find-interpreter -d 'Find interpreters from the host machine'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l skip-auditwheel -d 'Don\'t check for manylinux compliance'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l zig -d 'For manylinux targets, use zig to ensure compliance for the chosen manylinux version'
complete -c maturin -n "__fish_maturin_using_subcommand build" -s q -l quiet -d 'Do not print cargo log messages'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l all-features -d 'Activate all available features'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l no-default-features -d 'Do not activate the `default` feature'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l ignore-rust-version -d 'Ignore `rust-version` specification in packages'
complete -c maturin -n "__fish_maturin_using_subcommand build" -s v -l verbose -d 'Use verbose output (-vv very verbose/build.rs output)'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l frozen -d 'Require Cargo.lock and cache are up to date'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l locked -d 'Require Cargo.lock is up to date'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l offline -d 'Run without accessing the network'
complete -c maturin -n "__fish_maturin_using_subcommand build" -l future-incompat-report -d 'Outputs a future incompatibility report at the end of the build (unstable)'
complete -c maturin -n "__fish_maturin_using_subcommand build" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s r -l repository -d 'The repository (package index) to upload the package to. Should be a section in the config file' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l repository-url -d 'The URL of the registry where the wheels are uploaded to. This overrides --repository' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s u -l username -d 'Username for pypi or your custom registry' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s p -l password -d 'Password for pypi or your custom registry' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l compatibility -d 'Control the platform tag and PyPI compatibility' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s i -l interpreter -d 'The python versions to build wheels for, given as the executables of interpreters such as `python3.9` or `/usr/bin/python3.8`' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s b -l bindings -d 'Which kind of bindings to use' -r -f -a "pyo3\t''
pyo3-ffi\t''
cffi\t''
uniffi\t''
bin\t''"
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s o -l out -d 'The directory to store the built wheels in. Defaults to a new "wheels" directory in the project\'s target directory' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l auditwheel -d 'Audit wheel for manylinux compliance' -r -f -a "repair\t'Audit and repair wheel for manylinux compliance'
check\t'Check wheel for manylinux compliance, but do not repair'
skip\t'Don\'t check for manylinux compliance'"
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s j -l jobs -d 'Number of parallel jobs, defaults to # of CPUs' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l profile -d 'Build artifacts with the specified Cargo profile' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s F -l features -d 'Space or comma separated list of features to activate' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l target -d 'Build for the target triple' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l target-dir -d 'Directory for all generated artifacts' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s m -l manifest-path -d 'Path to Cargo.toml' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l color -d 'Coloring: auto, always, never' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l config -d 'Override a configuration value (unstable)' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s Z -d 'Unstable (nightly-only) flags to Cargo, see \'cargo -Z help\' for details' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l timings -d 'Timing output formats (unstable) (comma separated): html, json' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l sbom-include -d 'Additional SBOM files to include in the `.dist-info/sboms` directory. Can be specified multiple times' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l compression-method -d 'Zip compression method. Only Stored and Deflated are currently compatible with all package managers' -r -f -a "deflated\t'Deflate compression (levels 0-9, default 6)'
stored\t'No compression'
bzip2\t'BZIP2 compression (levels 0-9, default 6)'
zstd\t'Zstandard compression (supported from Python 3.14; levels -7-22, default 3)'"
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l compression-level -d 'Zip compression level. Defaults to method default' -r
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l debug -d 'Do not pass --release to cargo'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l no-strip -d 'Do not strip the library for minimum file size'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l no-sdist -d 'Don\'t build a source distribution'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l skip-existing -d 'Continue uploading files if one already exists. (Only valid when uploading to PyPI. Other implementations may not support this.)'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l non-interactive -d 'Do not interactively prompt for username/password if the required credentials are missing'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s f -l find-interpreter -d 'Find interpreters from the host machine'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l skip-auditwheel -d 'Don\'t check for manylinux compliance'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l zig -d 'For manylinux targets, use zig to ensure compliance for the chosen manylinux version'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s q -l quiet -d 'Do not print cargo log messages'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l all-features -d 'Activate all available features'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l no-default-features -d 'Do not activate the `default` feature'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l ignore-rust-version -d 'Ignore `rust-version` specification in packages'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s v -l verbose -d 'Use verbose output (-vv very verbose/build.rs output)'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l frozen -d 'Require Cargo.lock and cache are up to date'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l locked -d 'Require Cargo.lock is up to date'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l offline -d 'Run without accessing the network'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -l future-incompat-report -d 'Outputs a future incompatibility report at the end of the build (unstable)'
complete -c maturin -n "__fish_maturin_using_subcommand publish" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand list-python" -l target -r
complete -c maturin -n "__fish_maturin_using_subcommand list-python" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand list-python" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s b -l bindings -d 'Which kind of bindings to use' -r -f -a "pyo3\t''
pyo3-ffi\t''
cffi\t''
uniffi\t''
bin\t''"
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s E -l extras -d 'Install extra requires aka. optional dependencies' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s G -l group -d 'Install dependency groups defined in pyproject.toml (PEP 735)' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l pip-path -d 'Use a specific pip installation instead of the default one' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s j -l jobs -d 'Number of parallel jobs, defaults to # of CPUs' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l profile -d 'Build artifacts with the specified Cargo profile' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s F -l features -d 'Space or comma separated list of features to activate' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l target -d 'Build for the target triple' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l target-dir -d 'Directory for all generated artifacts' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s m -l manifest-path -d 'Path to Cargo.toml' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l color -d 'Coloring: auto, always, never' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l config -d 'Override a configuration value (unstable)' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s Z -d 'Unstable (nightly-only) flags to Cargo, see \'cargo -Z help\' for details' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l timings -d 'Timing output formats (unstable) (comma separated): html, json' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l compression-method -d 'Zip compression method. Only Stored and Deflated are currently compatible with all package managers' -r -f -a "deflated\t'Deflate compression (levels 0-9, default 6)'
stored\t'No compression'
bzip2\t'BZIP2 compression (levels 0-9, default 6)'
zstd\t'Zstandard compression (supported from Python 3.14; levels -7-22, default 3)'"
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l compression-level -d 'Zip compression level. Defaults to method default' -r
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s r -l release -d 'Pass --release to cargo'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l strip -d 'Strip the library for minimum file size'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l skip-install -d 'Skip installation, only build the extension module inplace'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s q -l quiet -d 'Do not print cargo log messages'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l all-features -d 'Activate all available features'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l no-default-features -d 'Do not activate the `default` feature'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l ignore-rust-version -d 'Ignore `rust-version` specification in packages'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s v -l verbose -d 'Use verbose output (-vv very verbose/build.rs output)'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l frozen -d 'Require Cargo.lock and cache are up to date'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l locked -d 'Require Cargo.lock is up to date'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l offline -d 'Run without accessing the network'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l future-incompat-report -d 'Outputs a future incompatibility report at the end of the build (unstable)'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -l uv -d 'Use `uv` to install packages instead of `pip`'
complete -c maturin -n "__fish_maturin_using_subcommand develop" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand sdist" -s m -l manifest-path -d 'The path to the Cargo.toml' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand sdist" -s o -l out -d 'The directory to store the built wheels in. Defaults to a new "wheels" directory in the project\'s target directory' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand sdist" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand sdist" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand init" -l name -d 'Set the resulting package name, defaults to the directory name' -r
complete -c maturin -n "__fish_maturin_using_subcommand init" -s b -l bindings -d 'Which kind of bindings to use' -r -f -a "pyo3\t''
cffi\t''
uniffi\t''
bin\t''"
complete -c maturin -n "__fish_maturin_using_subcommand init" -l mixed -d 'Use mixed Rust/Python project layout'
complete -c maturin -n "__fish_maturin_using_subcommand init" -l src -d 'Use Python first src layout for mixed Rust/Python project'
complete -c maturin -n "__fish_maturin_using_subcommand init" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand init" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand new" -l name -d 'Set the resulting package name, defaults to the directory name' -r
complete -c maturin -n "__fish_maturin_using_subcommand new" -s b -l bindings -d 'Which kind of bindings to use' -r -f -a "pyo3\t''
cffi\t''
uniffi\t''
bin\t''"
complete -c maturin -n "__fish_maturin_using_subcommand new" -l mixed -d 'Use mixed Rust/Python project layout'
complete -c maturin -n "__fish_maturin_using_subcommand new" -l src -d 'Use Python first src layout for mixed Rust/Python project'
complete -c maturin -n "__fish_maturin_using_subcommand new" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand new" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand generate-ci" -s m -l manifest-path -d 'Path to Cargo.toml' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand generate-ci" -s o -l output -d 'Output path' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand generate-ci" -l platform -d 'Platform support' -r -f -a "all\t'All'
manylinux\t'Manylinux'
musllinux\t'Musllinux'
windows\t'Windows'
macos\t'macOS'
emscripten\t'Emscripten'"
complete -c maturin -n "__fish_maturin_using_subcommand generate-ci" -l pytest -d 'Enable pytest'
complete -c maturin -n "__fish_maturin_using_subcommand generate-ci" -l zig -d 'Use zig to do cross compilation'
complete -c maturin -n "__fish_maturin_using_subcommand generate-ci" -l skip-attestation -d 'Skip artifact attestation'
complete -c maturin -n "__fish_maturin_using_subcommand generate-ci" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand generate-ci" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand upload" -s r -l repository -d 'The repository (package index) to upload the package to. Should be a section in the config file' -r
complete -c maturin -n "__fish_maturin_using_subcommand upload" -l repository-url -d 'The URL of the registry where the wheels are uploaded to. This overrides --repository' -r
complete -c maturin -n "__fish_maturin_using_subcommand upload" -s u -l username -d 'Username for pypi or your custom registry' -r
complete -c maturin -n "__fish_maturin_using_subcommand upload" -s p -l password -d 'Password for pypi or your custom registry' -r
complete -c maturin -n "__fish_maturin_using_subcommand upload" -l skip-existing -d 'Continue uploading files if one already exists. (Only valid when uploading to PyPI. Other implementations may not support this.)'
complete -c maturin -n "__fish_maturin_using_subcommand upload" -l non-interactive -d 'Do not interactively prompt for username/password if the required credentials are missing'
complete -c maturin -n "__fish_maturin_using_subcommand upload" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand upload" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and not __fish_seen_subcommand_from write-dist-info build-wheel write-sdist help" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and not __fish_seen_subcommand_from write-dist-info build-wheel write-sdist help" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and not __fish_seen_subcommand_from write-dist-info build-wheel write-sdist help" -f -a "write-dist-info" -d 'The implementation of prepare_metadata_for_build_wheel'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and not __fish_seen_subcommand_from write-dist-info build-wheel write-sdist help" -f -a "build-wheel" -d 'Implementation of build_wheel'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and not __fish_seen_subcommand_from write-dist-info build-wheel write-sdist help" -f -a "write-sdist" -d 'The implementation of build_sdist'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and not __fish_seen_subcommand_from write-dist-info build-wheel write-sdist help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l compatibility -d 'Control the platform tag and PyPI compatibility' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s i -l interpreter -d 'The python versions to build wheels for, given as the executables of interpreters such as `python3.9` or `/usr/bin/python3.8`' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s b -l bindings -d 'Which kind of bindings to use' -r -f -a "pyo3\t''
pyo3-ffi\t''
cffi\t''
uniffi\t''
bin\t''"
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s o -l out -d 'The directory to store the built wheels in. Defaults to a new "wheels" directory in the project\'s target directory' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l auditwheel -d 'Audit wheel for manylinux compliance' -r -f -a "repair\t'Audit and repair wheel for manylinux compliance'
check\t'Check wheel for manylinux compliance, but do not repair'
skip\t'Don\'t check for manylinux compliance'"
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s j -l jobs -d 'Number of parallel jobs, defaults to # of CPUs' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l profile -d 'Build artifacts with the specified Cargo profile' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s F -l features -d 'Space or comma separated list of features to activate' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l target -d 'Build for the target triple' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l target-dir -d 'Directory for all generated artifacts' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s m -l manifest-path -d 'Path to Cargo.toml' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l color -d 'Coloring: auto, always, never' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l config -d 'Override a configuration value (unstable)' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s Z -d 'Unstable (nightly-only) flags to Cargo, see \'cargo -Z help\' for details' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l timings -d 'Timing output formats (unstable) (comma separated): html, json' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l sbom-include -d 'Additional SBOM files to include in the `.dist-info/sboms` directory. Can be specified multiple times' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l compression-method -d 'Zip compression method. Only Stored and Deflated are currently compatible with all package managers' -r -f -a "deflated\t'Deflate compression (levels 0-9, default 6)'
stored\t'No compression'
bzip2\t'BZIP2 compression (levels 0-9, default 6)'
zstd\t'Zstandard compression (supported from Python 3.14; levels -7-22, default 3)'"
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l compression-level -d 'Zip compression level. Defaults to method default' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l metadata-directory -d 'The metadata_directory argument to prepare_metadata_for_build_wheel' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l strip -d 'Strip the library for minimum file size. Can be set to `true` or `false`, or used as a flag (`--strip` implies `true`)' -r -f -a "true\t''
false\t''"
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s f -l find-interpreter -d 'Find interpreters from the host machine'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l skip-auditwheel -d 'Don\'t check for manylinux compliance'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l zig -d 'For manylinux targets, use zig to ensure compliance for the chosen manylinux version'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s q -l quiet -d 'Do not print cargo log messages'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l all-features -d 'Activate all available features'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l no-default-features -d 'Do not activate the `default` feature'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l ignore-rust-version -d 'Ignore `rust-version` specification in packages'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s v -l verbose -d 'Use verbose output (-vv very verbose/build.rs output)'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l frozen -d 'Require Cargo.lock and cache are up to date'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l locked -d 'Require Cargo.lock is up to date'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l offline -d 'Run without accessing the network'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -l future-incompat-report -d 'Outputs a future incompatibility report at the end of the build (unstable)'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-dist-info" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l compatibility -d 'Control the platform tag and PyPI compatibility' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s i -l interpreter -d 'The python versions to build wheels for, given as the executables of interpreters such as `python3.9` or `/usr/bin/python3.8`' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s b -l bindings -d 'Which kind of bindings to use' -r -f -a "pyo3\t''
pyo3-ffi\t''
cffi\t''
uniffi\t''
bin\t''"
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s o -l out -d 'The directory to store the built wheels in. Defaults to a new "wheels" directory in the project\'s target directory' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l auditwheel -d 'Audit wheel for manylinux compliance' -r -f -a "repair\t'Audit and repair wheel for manylinux compliance'
check\t'Check wheel for manylinux compliance, but do not repair'
skip\t'Don\'t check for manylinux compliance'"
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s j -l jobs -d 'Number of parallel jobs, defaults to # of CPUs' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l profile -d 'Build artifacts with the specified Cargo profile' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s F -l features -d 'Space or comma separated list of features to activate' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l target -d 'Build for the target triple' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l target-dir -d 'Directory for all generated artifacts' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s m -l manifest-path -d 'Path to Cargo.toml' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l color -d 'Coloring: auto, always, never' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l config -d 'Override a configuration value (unstable)' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s Z -d 'Unstable (nightly-only) flags to Cargo, see \'cargo -Z help\' for details' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l timings -d 'Timing output formats (unstable) (comma separated): html, json' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l sbom-include -d 'Additional SBOM files to include in the `.dist-info/sboms` directory. Can be specified multiple times' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l compression-method -d 'Zip compression method. Only Stored and Deflated are currently compatible with all package managers' -r -f -a "deflated\t'Deflate compression (levels 0-9, default 6)'
stored\t'No compression'
bzip2\t'BZIP2 compression (levels 0-9, default 6)'
zstd\t'Zstandard compression (supported from Python 3.14; levels -7-22, default 3)'"
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l compression-level -d 'Zip compression level. Defaults to method default' -r
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l strip -d 'Strip the library for minimum file size. Can be set to `true` or `false`, or used as a flag (`--strip` implies `true`)' -r -f -a "true\t''
false\t''"
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s f -l find-interpreter -d 'Find interpreters from the host machine'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l skip-auditwheel -d 'Don\'t check for manylinux compliance'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l zig -d 'For manylinux targets, use zig to ensure compliance for the chosen manylinux version'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s q -l quiet -d 'Do not print cargo log messages'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l all-features -d 'Activate all available features'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l no-default-features -d 'Do not activate the `default` feature'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l ignore-rust-version -d 'Ignore `rust-version` specification in packages'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s v -l verbose -d 'Use verbose output (-vv very verbose/build.rs output)'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l frozen -d 'Require Cargo.lock and cache are up to date'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l locked -d 'Require Cargo.lock is up to date'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l offline -d 'Run without accessing the network'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l future-incompat-report -d 'Outputs a future incompatibility report at the end of the build (unstable)'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -l editable -d 'Build editable wheels'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from build-wheel" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-sdist" -l sdist-directory -d 'The sdist_directory argument to build_sdist' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-sdist" -s m -l manifest-path -d 'The path to the Cargo.toml' -r -F
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-sdist" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from write-sdist" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from help" -f -a "write-dist-info" -d 'The implementation of prepare_metadata_for_build_wheel'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from help" -f -a "build-wheel" -d 'Implementation of build_wheel'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from help" -f -a "write-sdist" -d 'The implementation of build_sdist'
complete -c maturin -n "__fish_maturin_using_subcommand pep517; and __fish_seen_subcommand_from help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c maturin -n "__fish_maturin_using_subcommand completions" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand completions" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -f -a "cc" -d '`zig cc` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -f -a "c++" -d '`zig c++` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -f -a "ar" -d '`zig ar` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -f -a "ranlib" -d '`zig ranlib` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -f -a "lib" -d '`zig lib` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -f -a "dlltool" -d '`zig dlltool` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and not __fish_seen_subcommand_from cc c++ ar ranlib lib dlltool help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from cc" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from cc" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from c++" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from c++" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from ar" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from ar" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from ranlib" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from ranlib" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from lib" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from lib" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from dlltool" -s v -l verbose -d 'Use verbose output'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from dlltool" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from help" -f -a "cc" -d '`zig cc` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from help" -f -a "c++" -d '`zig c++` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from help" -f -a "ar" -d '`zig ar` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from help" -f -a "ranlib" -d '`zig ranlib` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from help" -f -a "lib" -d '`zig lib` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from help" -f -a "dlltool" -d '`zig dlltool` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand zig; and __fish_seen_subcommand_from help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "build" -d 'Build the crate into python packages'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "publish" -d 'Build and publish the crate as python packages to pypi'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "list-python" -d 'Search and list the available python installations'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "develop" -d 'Install the crate as module in the current virtualenv'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "sdist" -d 'Build only a source distribution (sdist) without compiling'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "init" -d 'Create a new cargo project in an existing directory'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "new" -d 'Create a new cargo project'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "generate-ci" -d 'Generate CI configuration'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "upload" -d 'Upload python packages to pypi'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "pep517" -d 'Backend for the PEP 517 integration. Not for human consumption'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "completions" -d 'Generate shell completions'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "zig" -d 'Zig linker wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand help; and not __fish_seen_subcommand_from build publish list-python develop sdist init new generate-ci upload pep517 completions zig help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from pep517" -f -a "write-dist-info" -d 'The implementation of prepare_metadata_for_build_wheel'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from pep517" -f -a "build-wheel" -d 'Implementation of build_wheel'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from pep517" -f -a "write-sdist" -d 'The implementation of build_sdist'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from zig" -f -a "cc" -d '`zig cc` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from zig" -f -a "c++" -d '`zig c++` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from zig" -f -a "ar" -d '`zig ar` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from zig" -f -a "ranlib" -d '`zig ranlib` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from zig" -f -a "lib" -d '`zig lib` wrapper'
complete -c maturin -n "__fish_maturin_using_subcommand help; and __fish_seen_subcommand_from zig" -f -a "dlltool" -d '`zig dlltool` wrapper'
